; ESC1.ASM	Electronic Speed Control
; Written by 	Chuck McManis (http://www.mcmanis.com/chuck)
; This version	24-NOV-02
; Copyright (c) 2002 Charles McManis, All Rights Reserved.
; RCS Version
; $Id: esc1.asm,v 1.2 2002-11-28 22:49:31-08 chuck_mcmanis Exp chuck_mcmanis $
;
; Description:
;			This module converts the Servo Gizmo into a low
;		cost electronic speed control. Ideal for hooking
;		to a BASIC stamp, or other embedded microprocessor,
;		servo signals between 1000 and 2000 uS in length
;		are converted into a PWM waveform on the motor output.
;		There are 16 forward and 16 reverse speeds. The base
;		frequency of the PWM signal is controlled by the crystal.
;
; Things to do:
;	Track down issue with pulses out of range
;	Reset to Factory mode.
;
; Change Log:
;	24-NOV-02	Initial Development
;       26-NOV-02       Cleanup
;
; NOTICE: THIS CODE COMES WITHOUT WARRANTY OF ANY KIND EITHER
;         EXPRESSED OR IMPLIED. USE THIS CODE AT YOUR OWN RISK!
;         I WILL NOT BE HELD RESPONSIBLE FOR ANY DAMAGES, DIRECT 
;         OR CONSEQUENTIAL THAT YOU MAY EXPERIENCE BY USING IT.
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
        TITLE "ESC 1.0 - Electronic Speed Control"
        LIST P=PIC16F628, C=120, N=50, R=HEX
        include "P16F628.inc"
        include "inc\16bits.inc"	; 16 bit math
        include "inc\util.inc"		; Utility routines
        include "inc\sg.inc"		; ServoGizmo Definitions
       	__FUSES _CP_OFF&_XT_OSC&_WDT_ON&_LVP_OFF&_MCLRE_OFF
        
        ERRORLEVEL 2     
        

;
; These are the "factory" values for minimum, mid, and 
; maximum servo inputs. Followed by the scaling factor
; to convert a value between 0 and 500 to one between 0-200. 
;
        ORG     H'2100'
FACT_CAL:        
        DE      low D'1072', high D'1072'
        DE      low D'1522', high D'1522'
        DE      low D'1972', high D'1972' 
        DE      B'10010000'                     ; 2.25 in 2.6 fixed point
        DE      B'10010000'
;
; These are the User calibration values. They get over written
; during the calibration process.
;        
USER_CAL:
        DE      low D'1072', high D'1072'
        DE      low D'1522', high D'1522'
        DE      low D'1972', high D'1972' 
        DE      B'10010000'                     ; 2.25 in 2.6 fixed point
        DE      B'10010000'
        
        CBLOCK H'20'
            ; 
            ; Used by the Pulse Capture ISR
            ;
            LEADING:2
            CAPTURE:2
            ;
            ; Used by the divide routine...
            ;
            RESULT:2
            DIVISOR:2           ; Used in the interrupt
            DIVIDEND:2          ; Routine when calibrating
            DIV_COUNT
            ;
            ; Used in the Main loop for scaling and creating
            ; messages.
            ;
            SCALE_TMP
            CMD_TMP
            TEMP_VALUE:2
            R_MIN:2             ; Computed minimum and Maximum
            R_MAX:2
            ;
            ; Message passing mechanism between ISR's and MAIN
            BOOLS               ; Boolean values
            MODE                ; operating mode
            ;
            ; Used by the Timer 0 ISR to compute time outs
            ; Note: it is Tee-Zero-underscore-COUNT
            ;
            PWM_COUNT			; Current PWM count
            PWM_PERIOD			; Current PWM period
            P_COUNT				; Desired PWM count
            PWM_COMMAND			; Desired PWM command
            T0_COUNT:2			; 16 bit timeout
            ;
            ; Variables used in the Button State machine
            ;
            B_TIME:2            ; Keep track of time button is pressed
            B_STATE             ; Current state (0, 1, or 2)
            BUTT_COUNT:2        ; Blink counter (could be a byte)
            B_TMP:2
            ;
            ; EEPROM/Calibration data Keep together!!!
            ;
            DATA_START:0
            CAP_MIN:2           ; Minimum value captured
            CAP_MID:2           ; Neutral value (midpoint) captured
            CAP_MAX:2           ; Maximum value captured
            SCALE_FWD           ; Forward Scaling factor
            SCALE_REV           ; Reverse Scaling factor
            DATA_END:0
        ENDC
;
; Compute the size in bytes of the data we keep in the EEPROM
;        
DATA_SZ EQU  ( DATA_END - DATA_START )  ; Size of the data block            
        
;
; These commands sent to PORTA will cause either the FWD relay to
; energize or the REV relay to energize.
;
CMD_FWD   	EQU     H'04'
CMD_REV		EQU     H'08'
CMD_OFF 	EQU   	H'00'

P_PERIOD	EQU		D'16'		; 16 speeds

GUARD_BAND      EQU     15      ; 5 uS on either side of neutral
;
; Flag definitions for our flags values.
;
GOT_ONE EQU     0               ; Boolean "Got One"    
UI_MODE EQU     1               ; Calibrating    
VALID   EQU     2               ; Valid pulses received
UPDATE  EQU     3               ; Update of EEPROM required.
TIMER   EQU     4               ; Timer expired.
ACTIVE  EQU     5               ; Command is active

CMD_TIMEOUT     EQU     D'500' ; one second activation time

DELAYED EQU     0               ; Delayed activation mode
        
I_CNT   EQU     D'20'          ; Reset output if you get nothing
                                ; for 20 mS.
        ORG H'0000'
        GOTO    INIT            ; Let's get this puppy rolling
;
; Code Section
;
; The code section of the ISR lives at 0x0004. Its possible to put a 
; jump here however doing so adds two clocks of latency to servicing 
; any interrupt.
;       
        ORG     H'0004'         ; Interrupt service routine     
        ISR_ENTER               ; Enter the Interrupt Service routine
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;       I N T E R R U P T   S E R V I C E   R O U T I N E S 
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
        BTFSS   PIR1, TMR2IF    ; Timer 2 is the real time (1ms) Clock
        GOTO    ISR_1           ; If it hasn't fired, check the servo
        BCF     PIR1, TMR2IF    ; If we saw a timer 2 tick note it.
        CALL    BUTT_STATE      ; Manage the button UI
        ;
        ; Process the PWM function in interrupt time...
        ;
        DECFSZ	PWM_PERIOD,F	; Period counter
        GOTO	PWM_ACTION	; If its not zero continue
        MOVLW	P_PERIOD	; Put the PERIOD Constant in W
        MOVWF	PWM_PERIOD	; Put it in the PERIOD register
        MOVF	P_COUNT,W	; Get the desired count
        MOVWF	PWM_COUNT	; And store it in the count.
        MOVF	PWM_COMMAND,W	; Get the current command
        MOVWF	PORTA		; And activate it.
PWM_ACTION:
	MOVF	PWM_COUNT,W     ; Put the count in W
	BTFSC	STATUS,Z	; Check for non-zero
	GOTO	PWM_OFF
	BSF	PORTB, ENA_3N4	; Ensure this is "on"
	DECF	PWM_COUNT,F	; Decrement by 1
	GOTO	I_CONT		; Continue
PWM_OFF:
	BCF	PORTB, ENA_3N4	; Turn off the PWM
        ;
        ; Continuation point
        ;
I_CONT: 
	DEC16	T0_COUNT	; Manage the 20mS Timeout
	BTFSS	STATUS,Z	; If its not zero continue
	GOTO	ISR_1		; Continue
        ;
        ; Count underflows when we've hit this interrupt "n" times,
        ; where n is the number in COUNT. The count is reset to
        ; I_CNT whenever we receive a valid pulse.
        ;
        ; We treat it as if we received a 'go neutral' command from
        ; the receiver.
        ;
        MOVI16  I_CNT, T0_COUNT
        MOV16   CAP_MID, CAPTURE ; Neutral
        BSF     BOOLS, GOT_ONE
        BCF     BOOLS, VALID
        BCF     PORTB, SIG_LED	; LED management, turn off Signal LED
        BSF     BOOLS, TIMER    ; Then signal a timeout
        ;
        ; Process interrupts from the Input Capture/Compare pin
        ; (CCP1 on the 16F628)
        ;
ISR_1:  
        BTFSS   PIR1, CCP1IF    ; Check to see that CCP1 interrupted
        GOTO    ISR_2           ; If not continue
        BCF     PIR1, CCP1IF    ; Re-enable it
        BTFSS   CCP1CON, CCP1M0 ; Check for falling edge watch
        GOTO    FALL_EDGE       ; Go pick up the falling edge
        MOVF    CCPR1L,W        ; else store leading edge value
        MOVWF   LEADING         ; into 16 bit work LEADING
        MOVF    CCPR1H,W
        MOVWF   LEADING+1
        BCF     CCP1CON, CCP1M0 ; Now wait to capture the trailing edge
        GOTO    ISR_2           ; Exit the interrupt service routine
        
FALL_EDGE:
        BSF     CCP1CON, CCP1M0 ; Re-set for trailing edge capture
        MOVF    CCPR1L,W        ; Store the captured value into
        MOVWF   CAPTURE         ; CAPT_LO and ...
        MOVF    CCPR1H,W
        MOVWF   CAPTURE+1       ;             ... CAPT_HI
        ;
        ; 16 bit subtract 
        ;     CAPTURE = CAPTURE - LEAD
        ;
        SUB16   CAPTURE, LEADING
        BSF     BOOLS, GOT_ONE  ; Indicate we have a new sample.
        BSF     BOOLS, VALID    ; Indicate we're getting valid cmds
        BSF     PORTB, SIG_LED  ; LED Management, got a valid signal
        MOVI16  I_CNT, T0_COUNT	; Reset timeout count
ISR_2:  
        ISR_EXIT
	NOLIST        
        include "esc-lookup.asm"
        include "button.asm"                
	LIST
        PAGE
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;     I N I T I A L I Z A T I O N   S E C T I O N
;
; Initialization, this is the start address for initializing the PIC.
; All peripherals are initialized and any memory locations that need
; to be pre-initialized (to values or to zero) is done here.
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
INIT:
        CLRF    STATUS          ; Set Bank 0
        CLRF    B_STATE
        CLRF    BOOLS
        CLRF    PORTA           ; Clear PortA
        CLRF    PORTB           ; and clear PortB
        MOVLW   H'07'           ; Make PortA Digital I/O
        MOVWF   CMCON           ; By setting CMCON<0:3>
        
        ; * * * * * *
        ; * BANK 1 Operations
        ; * * * * * *
        BSF     STATUS,RP0      ; Set Bank 1
        MOVLW   B'0001010'      ; Set WDT prescaler to 8 (~144mS)
        MOVWF   OPTION_REG      ; Store it in the OPTION register
        CLRF    TRISA           ; Now A is all outputs
        CLRF    TRISB           ; B all outputs
        BSF     TRISB,BUTTON    ; Button S1 input
        BSF     TRISB,SERVO     ; CCP1 is an input.
        BSF     TRISA,W1        ; J1 input
        BSF     TRISA,W2        ; J2 input
        BSF     PIE1, CCP1IE    ; Enable interrupts from CCP1
        BSF     PIE1, TMR2IE    ; Enable interrupts from Timer 2
        MOVLW   D'250'          ; At 4Mhz this gives us 1mS interrupts.
        MOVWF   PR2             ; Store that in PR2
        
        ; * * * * * * * * * * *
        ; * BANK 0 Operations *
        ; * * * * * * * * * * *
        CLRF    STATUS          ; Back to BANK 0
        MOVLW   B'00000001'     ; Enable Timer 1 1:1 Prescale
        MOVWF   T1CON
        MOVLW   B'00000101'     ; Capture mode rising edge
        MOVWF   CCP1CON
        MOVLW   B'00000101'     ; Timer 2 on 4:1 prescaler, 1:1 Postscaler
        MOVWF   T2CON
        
        BSF     PORTB,CAL_LED   ; Turn On CAL
        CALL    IDLE_0          ; Initialize state machine.
        CLRF    BOOLS           ; Clear flags
        CLRF    MODE            ; Clear mode flags
        ;
        ; Load calibration data
        ;
        EEREAD  USER_CAL, DATA_START, DATA_SZ
        CLR16   CAPTURE
        MOVI16	I_CNT, T0_COUNT
        CLRF	P_COUNT
        BTFSC   PORTA, W2       ; Requesting a RESET to Factory
        GOTO    NO_RESET        ; Nope
        EEREAD  FACT_CAL, DATA_START, DATA_SZ
        BSF     BOOLS, UPDATE
;
; The last thing we do is enable interrupts 
;        
NO_RESET:
        BSF     INTCON, PEIE    ; Enable Peripheral Interrupts
        BSF     INTCON, GIE     ; Enable interrupts
        BCF     PORTB, CAL_LED
        PAGE

; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;     M A I N   P R O G R A M
;
; At this point interrrupts are enabled (and that is where most of
; the work is done. In the main routine we wait for the interrupt
; routines to activate various flags to tell us what to do. 
;
; The main loop implements a simple priority scheme, each action bit
; is checked in priority order, if it is set then that action 
; associated with the bit is called. Once finished the action
; returns to the head of the loop.
;
; Priority in the scheme is :
;		#1 UI Calibration - put the board in calibrate mode
;		#2 Time outs - Shut down the motor on timeout
;		#3 EEPROM Updates - store new calibration data.
;		#4 Servo Pulses - Process a servo pulse.
; 
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

MAIN:
        CLRWDT                  ; If we're here, we're cool
        BTFSC   BOOLS, UI_MODE  ; Are we doing Calibration?
        GOTO    MAIN            ; Yes so just loop here...
        BTFSC   BOOLS, UPDATE   ; Do we need to update EEPROM?
        GOTO    DO_UPDATE       ; Yes, so update EEPROM ...
        BTFSC   BOOLS, TIMER    ; Check for timer ticks
        GOTO    TIME_OUT        ; Process the time out counter
        BTFSC   BOOLS, GOT_ONE  ; Did we get a Servo Input
        GOTO    DO_CMD          ; Yes, so process that...
        GOTO    MAIN
        ;
        ; When a timeout occurs, we turn off the PWM
        ; function by reseting the P_COUNT to 0 and
        ; forcing the motors off.
        ; 
        ; Note that counts are cleared first, then the
        ; bit is reset to avoid a race condition where
        ; we turn off the bit, take an interrupt that
        ; turns it on again, and then reset the count.
        ; That would result in one extra cycle of PWM.
        ;
TIME_OUT:
        BCF     BOOLS, TIMER
        CLRF	P_COUNT			; Clear the PWM command
        CLRF	PWM_COUNT		; Clear the current count
        BCF		PORTB, ENA_3N4	; Turn off PWM
        GOTO    MAIN
        ;
        ; If updated, then store new values in the EEPROM.
        ;
DO_UPDATE:        
        BCF     BOOLS, UPDATE
        EEWRITE DATA_START, USER_CAL, DATA_SZ
        GOTO    MAIN
        
        ;
        ; Process a servo Command
        ;       Adjust it for guardband
        ;       Scale it between 0 and 200
        ;       Extract the command (fwd, rev, brake)
        ;
        ; scale it to 0 to 200 
        ;
DO_CMD:        
        BCF     BOOLS, GOT_ONE
        ;
        ; First Clip the value to CAP_MIN <= x <= CAP_MAX
        ;
        CMP16   CAPTURE, CAP_MAX        ; Is it greater than MAX?
        SKPC
        GOTO    NOT_MAX
        MOV16   CAP_MAX, CAPTURE        ; Use MAX instead
NOT_MAX:        
        CMP16   CAPTURE, CAP_MIN
        SKPNC
        GOTO    NOT_MIN
        MOV16   CAP_MIN, CAPTURE
NOT_MIN:
        MOV16   CAPTURE, TEMP_VALUE     ; May have been modified.
        SUB16   TEMP_VALUE, CAP_MID     ; Get magnitude from Neutral
        SKPC                            ; If carry set result was +
        GOTO    NEG_MAGNITUDE           ; Yes, so process it.
        MOVLW   CMD_FWD                 ; Forward command 
        MOVWF	CMD_TMP					; Put this in the command byte.
        MOVF    SCALE_FWD,W             ; Get forward scaling factor
        MOVWF   SCALE_TMP               ; and put it here for now.
        GOTO    FINISH_UP               ; 
NEG_MAGNITUDE:
        MOVLW   CMD_REV                 ; Its a reverse command
        MOVWF   CMD_TMP                 ; Put it in the packet
        MOVF    SCALE_REV,W             ; Get scaling factor
        MOVWF   SCALE_TMP               ; Store it here for now
        NEG16   TEMP_VALUE              ; Negate value 
FINISH_UP:
        CMPI16  TEMP_VALUE, GUARD_BAND  ; Guard band around 0
        SKPC                            ; Greater than guard band.
        GOTO    STOP_COMMAND            ; Yup, change it to a stop
        SUBI16   TEMP_VALUE, GUARD_BAND ; Subtract it from result.
        MOV16   TEMP_VALUE, DIVIDEND    ; Put in the dividend
        MOVF    SCALE_TMP,W             ; Get scaling factor
        MOVWF   DIVISOR                 ; Store in low byte of divisor
        CLRF    DIVISOR+1               ; Clear upper byte
        CALL    FP_DIVIDE               ; Do the divide.
        GOTO    DO_COMMAND				; Execute the command
        ;
        ; Braking behavior, if W1 is install we BRAKE
        ; otherwise we COAST when the command goes neutral.
        ;
STOP_COMMAND:
        BTFSS   PORTA, W1               ; Check to see if W1 installed
        GOTO    DO_BRAKE                ; Yup so don't coast.
        BCF    	PORTB, ENA_3N4	       	; ELSE Turn off PWM
        CLRF	P_COUNT		       	; Issue it
        GOTO	MAIN                    ; Wait for more commands
DO_BRAKE:        
        MOVLW   CMD_OFF                 ; Get stop command
        MOVWF   CMD_TMP                 ; Store it
        MOVLW   D'199'                  ; 50% duty cycle
        MOVWF   RESULT                  ; Store that in result
DO_COMMAND:
	BCF	STATUS,C       	        ; Clear carry
	RRF	RESULT,W             	; Rotate right, leave result in W
	DI
	CALL	GET_PWM		       	; Convert it to between 0 and period
	MOVWF	P_COUNT		       	; That is the new P_COUNT
	EI
	MOVF	CMD_TMP,W	       	; Get the command
	MOVWF	PWM_COMMAND	       	; Store it in the command
	; This will take effect on the _next_ cycle
	GOTO	MAIN
;
; FP_DIVIDE
;
; This does a special "fixed point" divide. It takes the
; value in DIVIDEND, shifts it six decimal places to the left
; and then divides by divisor. Thus the result is a floating point
; number with six significant fraction bits.
;
FP_DIVIDE:
        MOVLW   6               ; Shift by six bits
        MOVWF   DIV_COUNT       ; Store in count
FPD_0:  LSL16   DIVIDEND
        DECFSZ  DIV_COUNT,F
        GOTO    FPD_0
        ;
        ; ... now fall through to the 16 bit divide function
        ;                

;
; 16-bit x 16-bit unsigned divide. 
;
;       Divide two 16 bit numbers returning a 16 bit result
;       and a 16 bit remainder.
;
; DIVISOR (top part) - 16 bit variable
; DIVIDEND (bottom part) - 16 bit variable
; 
; Result is the a 16 bit result, the low byte is generally the
; interesting one.
;
; Don't divide by zero, its bad.
;
; Requires 7 bytes of RAM
;       RESULT:2
;       DIVISOR:2
;       DIVIDEND:2
;       DIV_COUNT
; DIV_COUNT is temporary for us...
; 
DIV16X16:
        CLR16   RESULT          ; Clear the result
        MOVF    DIVISOR,W       ; Check for zero
        IORWF   DIVISOR+1,W     ; 
        BTFSC   STATUS,Z        ; Check for zero
        RETLW   H'FF'           ; return 0xFF if illegal
        MOVLW   1               ; Start count at 1
        MOVWF   DIV_COUNT       ; Clear Count
D1:     BTFSC   DIVISOR+1,7     ; High bit set ?
        GOTO    D2              ; Yes then continue
        INCF    DIV_COUNT,F     ; Increment count
        LSL16   DIVISOR         ; Shift it left
        GOTO    D1
D2:     LSL16   RESULT          ; Shift result left
        SUB16   DIVIDEND, DIVISOR ; Reduce Divisor
        BTFSC   STATUS, C       ; Did it reduce?        
        GOTO    D3              ; No, so it was less than
        ADD16   DIVIDEND, DIVISOR ; Reverse subtraction
        GOTO    D4              ; Continue the process
D3:     BSF     RESULT,0        ; Yes it did, this gets a 1 bit
D4:     DECF    DIV_COUNT,F     ; Decrement N_COUNT
        BTFSC   STATUS,Z        ; If its not zero then continue
        GOTO    D5              ; Now we round the last bit.        
        LSR16   DIVISOR         ; Adjust divisor
        GOTO    D2              ; Next bit.
D5:     LSL16   DIVIDEND        ; Shift the dividend left by 1
        SUB16   DIVIDEND, DIVISOR ; See if that makes it larger than divisor
        BTFSS   STATUS,C         ;
        RETURN
        INC16   RESULT          ; Round up.        
        RETLW   0               ; Else return
                 
        END                      
