;
; UTIL.INC      - Utility Routines
; This Version    13-Apr-02
; RCS Version
; $Id: util.inc,v 1.0 2002-11-28 18:33:12-08 chuck_mcmanis Exp chuck_mcmanis $
;
        NOLIST
; Written by      Chuck McManis (http://www.mcmanis.com/chuck)
; Copyright (c) 2001 Charles McManis, All Rights Reserved
;
; Change Log:
;       12-Apr-02       Added EEREAD and EEWRITE, split off
;                       for the PIC1 project on the new ESC.
;       28-DEC-01       Added Digit conversion routines
;       22-DEC-01       Initially created with the WAIT
;                       macro for computing delays.
;
; NOTICE: THIS CODE COMES WITHOUT WARRANTY OF ANY KIND EITHER
;         EXPRESSED OR IMPLIED. USE THIS CODE AT YOUR OWN RISK!
;         I WILL NOT BE HELD RESPONSIBLE FOR ANY DAMAGES, DIRECT 
;         OR CONSEQUENTIAL THAT YOU MAY EXPERIENCE BY USING IT.
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;
; These variables are required by the utility routines to exist
; so they are declared in the include file.
;
        CBLOCK H'70'
            ISR_W               ; Storage for W
            ISR_FSR             ; Copy of the FSR
            ISR_STATUS          ; Copy of the Status register
            EE_SRC              ; EEPROM Source Pointer
            EE_DST              ; EEPROM Destination Pointer
            EE_LEN              ; EEPROM Byte count
            DLY_CNT             ; Delay counter
        ENDC

;
; Wait Macro
; 
; This macro comes from the "Advanced" PIC programming seminar
; and it will build a delay of 'n' machine cycles which are 
; 1/4 the clock frequency. So for a 4Mhz crystal a delay of
; 1 cycle is 4 ticks, or 1 instruction cycle which is 1uS. 
;
WAIT    MACRO   CYCLES
        nolist
X SET (CYCLES) % 4
        IF (X == 1) || (X == 3)
            NOP
        ENDIF
        
        IF (X == 2) || (X == 3)
            GOTO $+1
        ENDIF
        
X SET (CYCLES) / 4
        IF (X)
            IF (X == 256)
X SET 0
            ENDIF
            
            MOVLW   X
            ADDLW   -1
            SKPZ
            GOTO    $-2
        ENDIF
        list
        ENDM

;
; Microsecond Delay Loop
; 
; This function is designed to delay for 100uS times the number
; of times through the loop
; Once through :
;               1+91+2+1+2+1+2 = 100
;                   
; Twice through
;               1+91+2+1+1+2+96+1+2+1+2 = 200
;                            ******
; Thrice through
;               1+91+2+1+1+2+96+1+1+2+96+1+2+1+2 = 300
;                            ******** ******
; "N" times through (n * 100) cycles.
;
; NOTE: This will have to be changed when you change the 
; frequency from 4Mhz, but for the LAB-X3 it works great!
;
US_DELAY MACRO
        LOCAL   D1_LOOP, D2_LOOP
        
        MOVWF   DLY_CNT         ; Store delay count        (1)
        WAIT    91              ; Delay 95
        GOTO    D2_LOOP         ; Check to see if it was 1 (2)
D1_LOOP:
        WAIT    96              ;       (20)
D2_LOOP:        
        DECF    DLY_CNT,F       ; Subtract 1            (1)
        INCFSZ  DLY_CNT,W       ; Check for underflow   (2)
        GOTO    D1_LOOP          ; Not zero so loop      (2)
        NOP
        RETURN        
        ENDM

;
; Enable Interrupts
;
EI      MACRO
        BSF     INTCON, GIE
        ENDM

;
; Disable Interrupts
;
DI      MACRO
        BCF     INTCON, GIE
        ENDM        

;
; Written 12-Apr-2002, Chuck McManis
;
; Uses the temporary variable EE_LEN, EE_SRC, and EE_DST
; During read, Source is EEPROM and Destination is RAM
;
; Read a block of EEPROM and store it in RAM
; Assumes RAM block is in BANK0.
;                                  
EEREAD  MACRO   SRC_ADDR, DST_ADDR, LEN
        LOCAL   RLOOP
        MOVLW   LEN             ; Number of bytes to copy
        MOVWF   EE_LEN          ; Store the byte count.
        MOVLW   SRC_ADDR        ; Get the source address in EEPROM
        MOVWF   EE_SRC          ; Store it here.
        MOVLW   DST_ADDR        ; Get the destination address
        MOVWF   EE_DST          ; Store it here.
RLOOP:        
        MOVF    EE_DST,W        ; Get byte destination
        MOVWF   FSR             ; Store it in the FSR
        BSF     STATUS, RP0     ; Switch to bank 1 
        MOVF    EE_SRC,W        ; Address to read
        MOVWF   EEADR           ; Start address
        BSF     EECON1, RD      ; Read the byte
        MOVF    EEDATA, W       ; Assume a single cycle read?
        BCF     STATUS, RP0     ; Back to bank 0
        MOVWF   INDF            ; Store in Destination
        INCF    EE_SRC,F        ; Next source address
        INCF    EE_DST,F        ; Next destination address
        DECFSZ  EE_LEN,F        ; Decrement the count
        GOTO    RLOOP           ; Loop if not done.
        ENDM    

;
; Written: 14-Apr-2002, Chuck McManis
;
; Uses the temporary variable EE_LEN, EE_SRC, and EE_DST
; During write Source is RAM and Destination is EEPROM
;
; Note: Due to a bug in the 16F628 you can "overrun" the ability
; of the chip to write the EEPROM if you just monitor the WR bit
; in EECON1 (if you write immediately after it goes low the EEPROM
; may not be written.
;
; Thus this macro assumes that when it starts there is not a write
; currently in progress!
;
; Usage: EEWRITE <data bytes address (0-0xff)>, <eeprom address>, <len>
;
; On the 16F628, EE_LEN, EE_SRC, and EE_DST should be in the range
; 0x70 - 0x7F (mirrored on all ram banks)
;
;                                
EEWRITE MACRO   SRC_ADDR, DST_ADDR, LEN
        LOCAL   WLOOP
       
        MOVLW   LEN             ; Get length
        MOVWF   EE_LEN          ; Store it in count
        MOVLW   SRC_ADDR        ; Get the source address
        MOVWF   EE_SRC          ; Store in the source addr
        MOVLW   DST_ADDR        ; EEPROM Destination address
        MOVWF   EE_DST          ; Store it too
WLOOP:  
        BCF     PIR1, EEIF      ; Clear this flag...
        MOVF    EE_SRC,W        ; Source Address
        MOVWF   FSR             ; Store in FSR
        MOVF    INDF,W          ; Get a data byte
        BSF     STATUS, RP0     ; Switch to Bank 1
        MOVWF   EEDATA          ; Store the data byte in EE Data
        MOVF    EE_DST,W        ; Get the destination address
        MOVWF   EEADR           ; .. this byte goes here
        BSF     EECON1, WREN    ; Enable Writes to EEPROM
;       BTFSS   EECON1, WR      ; This should work but doesn't (see
;       GOTO    $-1             ; below) it only writes 2 of n bytes.        
        BCF     INTCON, GIE     ; Interrupts off.
        MOVLW   H'55'           ; Magic Sequence (Sequence Start)
        MOVWF   EECON2          ;       ... 
        MOVLW   H'AA'           ;       ... 
        MOVWF   EECON2          ;       ... 
        BSF     EECON1, WR      ; and Write it! (Sequence End)
        BSF     INTCON, GIE     ; Interrupts are allowed again.
        BCF     EECON1, WREN    ; Disable future writes.
        BCF     STATUS, RP0     ; back to bank 0
        BTFSS   PIR1, EEIF      ; Wait for it to finish, this slows
        GOTO    $-1             ;   ... us down! (bug! see above)
        INCF    EE_SRC, F       ; Add 1 to source and destination
        INCF    EE_DST, F       ;  ... addresses (no overlap)
        DECFSZ  EE_LEN,F        ; Decrement the length and ...
        GOTO    WLOOP           ; ... loop if non-zero.
        ENDM

; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;       I N T E R R U P T   S E R V I C E   M A C R O S
;
; ISR_ENTER is the "pre-amble" that stores W and STATUS safely away
; ISR_EXIT is the "post-amble" that restores W and STATUS and returns
; to the previous point of execution.
;
; ISR_ENTER takes 6 Tcy
; ISR_EXIT takes 8 Tcy
; Thus there is a built in 14 cycle overhead for interrupt services.
; Async interrupts can take 3 Tcy to dispatch and sync interrupts
; only take 2 Tcy to dispatch, so there is a 8 or 9 Tcy to first
; instruction of the ISR and a total of 17 Tcy (MAX) in overhead.
; 
; Interrupts are prioritized by the order in which they are tested
; for (polled). The general format is
;
;    ISR_n:
;       BTFSS   Reg, Flag
;       GOTO    ISR_n+1
;        ...
;    ISR_n+1:
;
; Thus there will be 3Tcy of latency for each ISR in the string.
; I'm using TMR0 as ISR_0 which means an interrupt faster than 
; about once every 100 clocks or so would be too fast (100 Tcy at
; 4 Mhz is 100 uS.)
;
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

ISR_ENTER       MACRO   
        MOVWF   ISR_W           ; Copy W to a temporary register
        SWAPF   STATUS,W        ; Swap Status Nibbles and move to W 
        MOVWF   ISR_STATUS      ; Copy STATUS to a temporary register
        CLRF    STATUS          ; Force Bank 0
        MOVF    FSR,W           ; Get the FSR
        MOVWF   ISR_FSR         ; Save it.
        ENDM

T0_CONST_4MHZ   EQU     D'133'  ; Constant for 1mS ticks at 4Mhz
;
; This macro handles the Timer0 interrupt for a 1Khz tick rate.
; Its not particularly accurate in the face of long term 
; interrupt masking but it gets the job done generally.
;             
T0_ISR  MACRO   NEXT_ISR
        ; ... test bit to see if Timer 0 interrupted...
        BTFSS   INTCON,T0IF     ; Did Timeer0 Overflow?
        GOTO    NEXT_ISR        ; No it didn't, so check next thing.
        ;
        ; Process Timer 0 Overflow Interrupt
        ;
        BCF     INTCON, T0IF    ; Clear Timer 0 interrupt
        MOVLW   T0_CONST_4MHZ   ; Reset counter to get 1 Khz interrupt
        MOVWF   TMR0            ; Store it.
        ENDM        

;
; This macro will initialize the Timer0 to operate as an interrupt
; source for the above routine.
;             
T0_INIT MACRO
        BSF     STATUS, RP0     ; Set Bank 1
        MOVLW   B'0000010'      ; Set TMR0 prescaler to 8
        MOVWF   OPTION_REG      ; Store it in the OPTION register
        BSF     INTCON, T0IE    ; Enable Timer 0 to interrupt
        BCF     INTCON, T0IF    ; Reset flag that indicates interrupt
        BCF     STATUS, RP0     ; Now in bank 0.
        ENDM
        

;
; Exit the interrupt service routine. 
; This involves recovering W and STATUS and then
; returning. Note that putting STATUS back automatically pops the bank
; back as well.
;               This takes 6 Tcy for a total overhead of 12 Tcy for sync
;               interrupts and 13 Tcy for async interrupts.
; 
ISR_EXIT        MACRO
        MOVF    ISR_FSR,W       ; Get FSR register
        MOVWF   FSR             ; put it back into FSR
        SWAPF   ISR_STATUS,W    ; Pull Status back into W
        MOVWF   STATUS          ; Store it in status 
        SWAPF   ISR_W,F         ; Prepare W to be restored
        SWAPF   ISR_W,W         ; Return it, preserving Z bit in STATUS
        RETFIE
        ENDM
        
;        
; PRESSED -- Jump to 'label' if a button is pressed.
;
PRESSED MACRO   label        
        BTFSS   PORTB, BUTTON
        GOTO    label
        ENDM
        
;        
; NOT_PRESSED -- Jump to 'label' if the button is not pressed.
;
NOT_PRESSED MACRO   label        
        BTFSC   PORTB, BUTTON
        GOTO    label
        ENDM

        
        LIST
